unit IWCompEdit;

{PUBDIST}

interface

uses
  {$IFDEF VSNET}
    System.ComponentModel, System.Drawing, IWNetComponent,
  {$ELSE}
  {$IFDEF Linux}
  IWCLXComponent,
  {$ELSE}
  IWVCLComponent,
  {$ENDIF}
  {$IFDEF Linux}QForms, {$ELSE}Forms, {$ENDIF}
  {$ENDIF}
  {$IFDEF CLR}
    IWNetFileParser,
  {$ELSE}
    IWFileParser,
  {$ENDIF}
  {$IFDEF Linux}QGraphics, {$ELSE}Graphics, {$ENDIF}
  Classes,
  IWControl,
  IWBaseControl, IWTypes, IWScriptEvents, IWHTMLTag, IWColor,
  IWRenderContext, IWBaseInterfaces, IWCompButton;

type
  TIWCustomEdit = class(TIWCustomControl, IIWInputControl, IIWSubmitControl)
  protected
    FSubmitParam : String;  
    FAlignment: TAlignment;
    FBGColor: TIWColor;
    FFocusColor: TIWColor;
    FMaxLength: Integer;
    FOnSubmit: TNotifyEvent;
    FPasswordPrompt: Boolean;
    FRequired: Boolean;
    FReadOnly: Boolean;
    FTagType: string;
    //
    {$IFDEF VSNET}
    function getText: string; reintroduce;virtual;
    procedure setText(AValue: string); reintroduce;virtual;
    {$ENDIF}
    procedure DoSubmit;
    procedure HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents); override;
    procedure SetValue(const AValue: string); virtual;
    procedure Submit(const AValue: string); override;
    procedure SetPasswordPrompt(const AValue: Boolean);
    procedure SetBGColor(const Value: TIWColor);
    procedure SetAlignment(const Value: TAlignment);
    procedure SetFocusColor(const Value: TIWColor);
    procedure SetMaxLength(const AValue:integer);
    procedure SetReadOnly(const AValue:boolean);
    procedure SetRequired(const AValue:boolean);
    procedure InitControl; override;
    function FormHasOnDefaultActionSet:boolean;
    function get_ShouldRenderTabOrder: boolean;override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    function RenderStyle(AContext: TIWBaseHTMLComponentContext): string; override;
    property PasswordPrompt: Boolean read FPasswordPrompt write SetPasswordPrompt;
    function GetSubmitParam : String;
    {$IFDEF VSNET}
    property Text: string read getText write setText;
    {$ENDIF}
  published
    property Alignment: TAlignment read FAlignment write SetAlignment;

    property BGColor: TIWColor read FBGColor write SetBGColor;
    property FocusColor: TIWColor read FFocusColor write SetFocusColor;
    property DoSubmitValidation;
    property Editable;
    property NonEditableAsLabel;

    {$IFDEF CLR}
    property WebFont;
    {$ELSE}
    property Font;
    {$ENDIF}
    property ExtraTagParams;
    property FriendlyName;

    property MaxLength: Integer read FMaxLength write SetMaxLength;

    property ReadOnly: Boolean read FReadOnly write SetReadOnly;

    property Required: Boolean read FRequired write SetRequired;
    property ScriptEvents;
    property TabOrder;

    property OnSubmit: TNotifyEvent read FOnSubmit write FOnSubmit;
  end;

{$IFDEF VSNET}
{$R icons\Atozed.Intraweb.TIWEdit.bmp}
  TIWEdit = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWEdit), 'TIWEdit.bmp')]
{$ENDIF}
  TIWEdit = class(TIWCustomEdit)
  public
    procedure Clear;
  published
    property Enabled;
    property PasswordPrompt;

    property Text;
  end;

  TReceivedFileEvent = procedure(ASender: TObject; const AFilename: string) of object;

  TIWFileButtonPosition = (ubpBelowEdit, ubpNextToBrowse);

  TIWCustomFile = class(TIWCustomControl, IIWInputControl, IIWSubmitControl)
  protected
    FSubmitParam : String;
    FBGColor: TIWColor;
    FFrameBGColor: TIWColor;
    FContentType: string;
    FFileName: string;
    FButtonHeight: Integer;

    FFile: THTTPFile;

    function GetContentType: string;
    function GetSize: Integer;
    procedure SetValue(const AValue: string); virtual;
    procedure SetFileValue(AFile: THTTPFile); override;

    procedure InitControl; override;
    procedure Submit(const AValue: string); override;
    procedure DoUploadButtonChange(Sender: TObject);
    procedure SetBGColor(const Value: TIWColor);
    procedure SetFrameBGColor(const Value: TIWColor);

{$IFDEF CLR}
    strict protected
{$ELSE}
  protected
{$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    function RenderStyle(AContext: TIWBaseHTMLComponentContext): string; override;
    function GetSubmitParam : String;
    procedure SaveToFile(AFilename: string = '');

    procedure SaveToStream(AStream: TStream);

    property ContentType: string read GetContentType;

    property Filename: string read FFileName;
    property ButtonHeight: Integer read FButtonHeight write FButtonHeight;
    property BGColor: TIWColor read FBGColor write SetBGColor;
    property FrameBGColor: TIWColor read FFrameBGColor write SetFrameBGColor;
  published
    property TabOrder;
  end;
  
{$IFDEF VSNET}
{$R icons\Atozed.Intraweb.TIWFile.bmp}
  TIWFile = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWFile), 'TIWFile.bmp')]
{$ENDIF}
  TIWFile = class(TIWCustomFile)
  published
    property Enabled;
    property FriendlyName;
    {$IFDEF CLR}
    property WebFont;
    {$ELSE}
    property Font;
    {$ENDIF}
    property BGColor;
    property ButtonHeight;
    property FrameBGColor;
  end;

implementation

uses
  SysUtils, 
  {$IFDEF VSNET}
  IWNetBaseControl,
  {$ENDIF}
  IWAppForm, IWApplication, IWBaseForm, IWCompLabel, IWResourceStrings, IWServerControllerBase,
  SWSystem, SWStrings, IWGlobal, IWForm, IWRegion, IWStreams,
  IWBaseHTMLControl, IWException;

{$IFDEF VSNET}
function TIWCustomEdit.getText: string;
begin
  Result := Caption;
end;

procedure TIWCustomEdit.setText(AValue: string);
begin
  Caption := AValue;
  TIWNETBaseControl(Self).Invalidate;
end;
{$ENDIF}

procedure TIWCustomEdit.SetValue(const AValue: string);
begin
  if RequiresUpdateNotification(Parent) then begin
    UpdateNotifiedInterface(Parent).NotifyUpdate(Self,AValue);
  end;
  if Text<>AValue then begin
    Text := AValue;
    Invalidate;
  end;
end;

function TIWCustomEdit.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

function TIWCustomEdit.get_ShouldRenderTabOrder: boolean;
begin
  result := Editable or (NonEditableAsLabel = false);
end;

function TIWCustomEdit.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LLabel: TIWLabel;
begin
  result := nil;
  if Editable or (NonEditableAsLabel = false) then begin
    Result := TIWHTMLTag.CreateTag('INPUT'); try
      Result.AddStringParam('TYPE', FTagType);
      Result.AddStringParam('NAME', HTMLName);
      if (not Editable) and (NonEditableAsLabel = false) then begin
        Result.AddBoolParam('DISABLED', true);
        Result.AddBoolParam('READONLY', true);
      end;
      if not Enabled then begin
        Result.AddBoolParam('DISABLED', True);
      end;
      if MaxLength > 0 then begin
        Result.AddIntegerParam('MAXLENGTH', MaxLength);
      end;
      Result.AddStringParam('VALUE', TextToHTML(Text));
      Result.Add(iif(ReadOnly or IsDesignMode, 'READONLY'));
      if Required then begin
        (AContext as TIWComponent40Context).AddValidation(HTMLName + 'IWCL != null && ' + HTMLName + 'IWCL.value.length==0'
          , Format(RSValidationRequeredField, [FriendlyName]));
      end;
    except FreeAndNil(Result); raise; end;
 end else begin
    LLabel := TIWLabel.Create{$IFNDEF VSNET}(Self){$ENDIF}; try
      LLabel.Name := Name;
      LLAbel.Alignment := Alignment;
      if FPasswordPrompt then begin
        LLabel.Caption := StringOfChar('*', Length(Text));
      end else begin
        LLabel.Caption := Text;
      end;
      LLabel.WebFont.Assign(WebFont);
      LLabel.Width := Width;
      LLabel.Height := Height;
      Result := LLabel.RenderHTML(AContext);
    finally FreeAndNil(LLabel); end;
  end; 
end;

function TIWCustomEdit.FormHasOnDefaultActionSet:boolean;
var
  LForm: TIWBaseForm;
begin
  Result := false;
  LForm := OwnerForm;
  if Assigned(LForm) then begin
    if LForm is TIWForm then begin
      result := Assigned( TIWForm(LForm).OnDefaultAction );
    end;
  end;
end;

procedure TIWCustomEdit.DoSubmit;
begin
  if Assigned(OnSubmit) then begin
    OnSubmit(Self);
  end else if FormHasOnDefaultActionSet then begin
    TIWForm(OwnerForm).OnDefaultAction(Self);
  end;
end;

procedure TIWCustomEdit.Submit(const AValue: string);
begin
  FSubmitParam := AValue;
  DoSubmit;
end;

function TIWCustomEdit.RenderStyle(AContext: TIWBaseHTMLComponentContext): string;
begin
  Result := inherited RenderStyle(AContext)
    + iif(toTColor(WebFont.Color) <> clNone, 'color:' + ColorToRGBString(WebFont.Color) + ';');
  if toTColor(BGColor) <> clNone then begin
    Result := Result + 'background-color:' + ColorToRGBString(BGColor) + ';';
  end;
  case Alignment of
    taLeftJustify: Result := Result + 'text-align:left;';
    taCenter: Result := Result + 'text-align:center;';
    taRightJustify: Result := Result + 'text-align:right;';
  end;
end;

procedure TIWCustomEdit.HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents);
begin
  inherited HookEvents(AContext, AScriptEvents);
  if Assigned(OnSubmit) or (FormHasOnDefaultActionSet and ( ClassName <> 'TTIWAdvEdit' )) then begin
    AScriptEvents.HookEvent('OnKeyPress', 'var res = CheckReturnKey('
      + iif(AContext.WebApplication.Browser = brIE, 'event.keyCode', 'event.which')
      + ', ''' + HTMLName + ''', ' + iif(DoSubmitValidation, 'true', 'false') + ')' + EOL
      // + iif(AContext.WebApplication.Browser <> brIE, 'if (!res) { noSecondSubmit = false; }'+ EOL)
      + 'return res;');
  end;
  if (toTColor(FocusColor) <> clNone) and Editable then begin
    AScriptEvents.HookEvent('OnFocus', 'this.style.backgroundColor=''' + ColorToRGBString(FocusColor) + '''; return true;');
    AScriptEvents.HookEvent('OnBlur', 'this.style.backgroundColor=''' + ColorToRGBString(BGColor) + '''; return true;');
  end;
end;

procedure TIWCustomEdit.InitControl;
begin
  inherited;
  FFocusColor := fromTColor(clNone);
  FBGColor := fromTColor(clNone);
  FNeedsFormTag := True;
  FTagType := 'TEXT';
  Height := 21;
  Width := 121;
  FAlignment := taLeftJustify;
  set_RenderSize(true);
  FCanReceiveFocus := True;
end;


procedure TIWCustomEdit.SetFocusColor(const Value: TIWColor);
begin
  if FFocusColor <> Value then
  begin
    FFocusColor := Value;
    Invalidate;
  end;
end;

procedure TIWCustomEdit.SetMaxLength(const AValue:integer);
begin
  if FMaxLength <> AValue then
  begin
    FMaxLength := AValue;
    Invalidate;
  end;
end;

procedure TIWCustomEdit.SetReadOnly(const AValue:boolean);
begin
  if FReadOnly <> AValue then
  begin
    FReadOnly := AValue;
    Invalidate;
  end;
end;

procedure TIWCustomEdit.SetRequired(const AValue:boolean);
begin
  if FRequired <> AValue then
  begin
    FRequired := AValue;
    Invalidate;
  end;
end;




{ TIWCustomFile }

function TIWCustomFile.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

function TIWCustomFile.GetSize: Integer;
begin
  result := 0;
  if Assigned(FFile) then
    Result := FFile.Stream.Size;
end;

function TIWCustomFile.GetContentType: string;
begin
  Result := '';
  if Assigned(FFile) then
    Result := FFile.ContentType;
end;

procedure TIWCustomFile.SaveToFile(AFilename: string = '');
begin
  if AFileName = '' then begin
    AFileName := Filename;
  end;
  if Assigned(FFile) then begin
    FFile.SaveToFile(AFileName);
  end;
end;

procedure TIWCustomFile.SaveToStream(AStream: TStream);
begin
  if Assigned(FFile) then begin
    FFile.SaveToStream(AStream);
  end;
end;

procedure TIWCustomEdit.SetPasswordPrompt(const AValue: Boolean);
begin
  FTagType := IIF(AValue, 'PASSWORD', 'TEXT');
  FPasswordPrompt := AValue;
end;

procedure TIWCustomEdit.SetBGColor(const Value: TIWColor);
begin
  FBGColor := Value;
  Invalidate;
end;

procedure TIWCustomEdit.SetAlignment(const Value: TAlignment);
begin                                         
  if FAlignment <> Value then begin
    FAlignment := Value;
    Invalidate;
  end;
end;

procedure TIWCustomFile.SetValue(const AValue: string);
begin
  if RequiresUpdateNotification(Parent) then begin
    UpdateNotifiedInterface(Parent).NotifyUpdate(Self,AValue);
  end;
  FFilename := AValue;
end;

function TIWCustomFile.RenderHTML(
  AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LFrameBody: TIWHTMLTag;
  LUploadForm: string;
  LAction: string;
  LBGColor: IWColor.TIWColor;
  LTmpRenderStream: TIWRenderStream;
begin
  EIWException.IfTrue((OwnerForm is TIWAppForm) and (TIWAppForm(OwnerForm).UpdateMode = umPartial),'Can''t use TIWFile when using umPartial.');
  EIWException.IfTrue((OwnerForm<>nil) and (OwnerForm.FormType = ftPage),'Can''t use TIWFile in Page-Mode.');

  Result := TIWHTMLTag.CreateTag('IFRAME');
  try
    LUploadForm := GServerController.NewCacheFile('html', True);

    with Result do
    begin
      AddStringParam('SRC', AContext.WebApplication.UserCacheURL + ExtractFilename(LUploadForm));
      AddStringParam('frameborder', '0');
      AddStringParam('scrolling', 'no');
      addintegerparam('width', width);
    end;

    LFrameBody := TIWHTMLTag.CreateTag('body');
    try
      with LFrameBody do
      begin
        AddStringParam('TOPMARGIN', '0');
        AddStringParam('LEFTMARGIN', '0');
        if FFrameBGColor <> fromTColor(clNone) then begin
          AddStringParam('BGCOLOR', ColorToRGBString(FFrameBGColor));
        end else begin
          LBGColor := fromTColor(clNone);
          if Parent is TIWForm then begin
            LBGColor := (Parent as TIWForm).BGColor;
          end else if Parent is TIWRegion then begin
            LBGColor := (Parent as TIWRegion).WebColor;
          end;
          if LBGColor <> fromTColor(clNone) then begin
            if (AContext.WebApplication.Browser = brIE) and
               (FFrameBGColor = fromTColor(clNone)) and
               (LBGColor <> fromTColor(clNone)) then begin
              AddStringParam('BGCOLOR', ColorToRGBString(LBGColor));
            end;
          end;
        end;
      end;
      with LFrameBody.Contents.AddTag('form') do begin
        AddStringParam('method', 'POST');
        AddStringParam('enctype', 'multipart/form-data');
        AddStringParam('name', 'HiddenSubmitForm');
        if not IsDesignMode then begin
          AddStringParam('onsubmit', 'IWTop().GSubmitting = true; IWTop().SetBusy(true);');
        end;
        with Contents.AddTag('input') do
        begin
          if not Enabled then begin
            AddBoolParam('DISABLED', True);
          end;
          AddStringParam('type', 'file');
          AddStringParam('name', HTMLName);
          AddStringParam('style', WebFont.FontToStringStyle(AContext.WebApplication.Browser) +
           iif(RenderSize, 'height:' + IntToStr(FButtonHeight) + '; width:' + IntToStr(Width) + ';', ''));
          if FBGColor <> fromTColor(clNone) then
          begin
            Params.Values['style'] := Params.Values['style'] + 'background-color:' + ColorToRGBString(FBGColor);
          end;
          if IsDesignMode or not Enabled then begin
            Add('DISABLED');
          end;
        end;
        with Contents.AddTag('INPUT') do
        begin
          AddStringParam('TYPE', 'HIDDEN');
          AddStringParam('NAME', 'IW_ActionParam');
          AddStringParam('value', HTMLName);
        end;

        with Contents.AddTag('INPUT') do
        begin
          AddStringParam('TYPE', 'HIDDEN');
          AddStringParam('NAME', 'IW_MainPageSubmitControl');
          AddStringParam('value', '');
        end;

        with Contents.AddTag('INPUT') do
        begin
          AddStringParam('TYPE', 'HIDDEN');
          AddStringParam('NAME', 'IW_MainPageSubmitControlParam');
          AddStringParam('value', '');
        end;

        if not IsDesignMode then begin
          with AContext.WebApplication do
            case GServerController.SessionTrackingMethod of
              tmCookie: begin
                  LAction := AppURLBase + '/upload/';
                  with Contents.AddTag('INPUT') do
                  begin
                    AddStringParam('TYPE', 'HIDDEN');
                    AddStringParam('NAME', 'IW_TrackID_');
                    AddIntegerParam('VALUE', TrackID);
                  end;
                end;
              tmURL: begin
                  LAction := AppURLBase + '/upload/' + IntToStr(TrackID) + '/' + AppID;
                  Invalidate;
                end;
              tmHidden: begin
                  LAction := AContext.WebApplication.AppURLBase + '/upload';
                  with Contents.AddTag('INPUT') do
                  begin
                    AddStringParam('TYPE', 'HIDDEN');
                    AddStringParam('NAME', 'IW_SessionID_');
                    AddStringParam('VALUE', AppID);
                  end;
                  with Contents.AddTag('INPUT') do
                  begin
                    AddStringParam('TYPE', 'HIDDEN');
                    AddStringParam('NAME', 'IW_TrackID_');
                    AddIntegerParam('VALUE', TrackID);
                  end;
                  Invalidate;
                end;
            end;
          AddStringParam('action', LAction);
        end;
      end;
      LTmpRenderStream:=TIWRenderStream.Create;
      try
        LTmpRenderStream.WriteString('<HTML><HEAD></HEAD>');
        LFrameBody.Render(LTmpRenderStream);
        LTmpRenderStream.WriteString('</HTML>');
        with TFileStream.Create(LUploadForm, fmCreate or fmShareDenyWrite) do try
          CopyFrom(LTmpRenderStream,0);
        finally
          Free;
        end;
      finally
        LTmpRenderStream.free;
      end;
    finally
      LFrameBody.Free;
    end;
  except
    FreeAndNil(Result);
  end;
end;

procedure TIWCustomFile.Submit(const AValue: string);
begin
  FSubmitParam := AValue;
  // Does nothing just to protect from exception in ExecuteForm
end;

{ TIWEdit }

procedure TIWEdit.Clear;
begin
  Text := '';
end;

procedure TIWCustomFile.InitControl;
begin
  inherited;
  Width := 260;
  Height := 80;
  FFile := nil;
  set_RenderSize(true);
  ControlEncode := ceMPFormData;
  FBGColor := fromTColor(clNone);
  FFrameBGColor := fromTColor(clNone);
  FButtonHeight := 21;
end;

procedure TIWCustomFile.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FFile);
  inherited;
end;

function TIWCustomFile.RenderStyle(
  AContext: TIWBaseHTMLComponentContext): string;
begin
  result := inherited RenderStyle(AContext);
  result := result + 'border-width:0;border-style:none;overflow:hidden;';
  if toTColor(FBGColor) <> clNone then begin
    Result := Result + 'background-color: ' + ColorToRGBString(FBGColor) + ';';
  end;
end;

procedure TIWCustomFile.SetFileValue(AFile: THTTPFile);
begin
  FreeAndNil(FFile);
  if Assigned(AFile) then begin
    FFile := THTTPFile.Create(AFile);
  end;
end;

procedure TIWCustomFile.DoUploadButtonChange(Sender: TObject);
begin
  Invalidate;
end;

procedure TIWCustomFile.SetFrameBGColor(const Value: TIWColor);
begin
  FFrameBGColor := Value;
  Invalidate;
end;

procedure TIWCustomFile.SetBGColor(const Value: TIWColor);
begin
  if FBGColor <> Value then
  begin
    FBGColor := Value;
    Invalidate;
  end;
end;




end.
